/*
      (C) Copyright 1990-1991 by Autodesk, Inc.

******************************************************************************
*                                                                            *
* The information contained herein is confidential, proprietary to Autodesk, *
* Inc., and considered a trade secret as defined in section 499C of the      *
* penal code of the State of California.  Use of this information by anyone  *
* other than authorized employees of Autodesk, Inc. is granted only under a  *
* written non-disclosure agreement, expressly prescribing the scope and      *
* manner of such use.                                                        *
*                                                                            *
******************************************************************************/

#ifndef LINALGHPP
#define LINALGHPP
#define BORLAND
#define NO_IOSTREAM

#ifndef NO_IOSTREAM
#ifdef BORLAND
#include <iostream.h>
#else
#include <stream.hpp>
#endif // not BORLAND
#endif // NO_IOSTREAM	

#include <math.h>
#include "grtypes.h"

#define X 0
#define Y 1
#define Z 2

#define EPSI 1e-6
#define REQ(x, y) (fabs((x) - (y)) < 1e-3)
/*
	linalg.hpp -- class declarations for vector and matrix manipulations

	Four classes are supported in this module:
		CyVector3 - an ordered triple representing direction and
			magnitude in 3-space
		CyVertex3 - an ordered triple representing a point in 3-space
		CyMatrix - a 4*4 array representing a rigid motion
		CyQuaternion - Four Real coefficients or
			A Real and a CyVector3 defining an orientation in 3-space

	Several operators are overloaded:
		CyVector3 * CyVector3 -> CyVector3   - cross product
		CyVector3 % CyVector3 -> Real      - dot product
		CyVector3 + CyVector3 -> CyVector3   - vector addition
		CyVector3 - CyVector3 -> CyVector3   - vector subtraction
		CyVector3 + CyVertex3 -> CyVertex3   - vertex translation
		CyVector3 = CyVector3 -> CyVector3   - assignment
		-CyVector3                       - unary minus
		CyVector3 * Real                 - scalar multiple
#ifndef NO_IOSTREAM
		ostream << CyVector3             - output as [x, y, z]
#endif //
		CyVector3[i]                     - access x, y, or z component

		CyVertex3 + CyVector3 -> CyVertex3   - vertex translation
		CyVertex3 - CyVector3 -> CyVertex3   - vertex translation
		CyVertex3 - CyVertex3 -> CyVector3   - point to point delta
		CyVertex3 = CyVertex3 -> CyVertex3   - assignment
		CyVertex3 * Real                 - scalar weight multiple
		CyVertex3 + CyVertex3 -> CyVertex3   - for averaging vertices.
#ifndef NO_IOSTREAM
		ostream << CyVertex3             - output as (x, y, z)
#endif //
		CyVertex3[i]                     - access x, y, or z component

		CyMatrix * CyMatrix -> CyMatrix      - matrix composition
		CyMatrix * CyVector3 -> CyVector3    - transform vector
		CyMatrix * CyVertex3 -> CyVertex3    - transform vertex

		CyQuaternion = CyQuaternion -> CyQuaternion - assignment
		CyQuaternion + CyQuaternion -> CyQuaternion - addition
		CyQuaternion - CyQuaternion -> CyQuaternion - subtraction
		CyQuaternion * CyQuaternion -> CyQuaternion - multiplication
		CyQuaternion % CyQuaternion -> CyQuaternion - dot product
*/

//
// VECTOR class 
// 
class CyVector3
{
protected:
	friend class CyVertex3;
	friend class CyMatrix;
	Real v[3];
public:
	CyVector3(Real,Real,Real);		// constructor with initialization
	CyVector3();				// construct zero vector
	CyVector3(CyVector3&);			// copy initializer
	CyVector3 operator+=(CyVector3&);	// assignment with addition
	CyVector3 operator-=(CyVector3&);	// assignment with subtract
	CyVector3 operator*=(CyVector3&);	// assignment with cross product
	void Print();				// print
	void Set(Real x,Real y,Real z);	// establish values

        Real& operator[](int i)			// index access
	{
           return v[i];
        }

	// unary ops
	CyVector3 operator-();			// invert
	Real Mag();				// length (magnitude)
	CyVector3& Unify();			// convert to unit length
	Real MagUnify(); // convert to unit length, and return old length.

	// binary ops
	CyVector3 operator+(CyVector3&);	// add 
	CyVector3 operator-(CyVector3&);	// subtract
	CyVector3 operator*(CyVector3&);	// cross product
	Real operator%(CyVector3&);		// dot product
	class CyVertex3 operator+(class CyVertex3&);	// vector + vertex
	Real Dot(CyVector3&);				// dot-product

	// polymorphic ops (as friends due to need to commute)
	friend CyVector3 operator*(CyVector3&,Real);
	friend CyVector3 operator*(Real,CyVector3&);
	friend CyVector3 operator/(CyVector3&,Real);
	friend CyVector3 operator/(Real,CyVector3&);
#ifndef NO_IOSTREAM
	friend ostream& operator<<(ostream& s, CyVector3& v);
#endif //
	// packetizers
	void EncodeToPkt(uchar*, int&);
	void DecodeFromPkt(uchar*, int&);
	int BytesNeededToEncode();
};

//
// Vertex class
//
class CyVertex3
{
protected:
	friend class CyVector3;
	friend class CyMatrix;
	Real v[3];
public:
	CyVertex3(Real,Real,Real);		// constructor with initialization
	CyVertex3();				// construct zero vector
	CyVertex3(CyVertex3&);			// copy initializer
	CyVertex3 operator+=(CyVector3&);	// assignment with vector offset
	void Print();                           // Print.
	void Set(Real x,Real y,Real z);		// establish values

        Real& operator[](int i)			// index access
	{
               return v[i];
        }

	// binary ops
	CyVector3 operator-(CyVertex3&);	// subtract
	CyVertex3 operator+(CyVector3&);	// vertex + vector
	CyVertex3 operator-(CyVector3&);	// vertex - vector
	CyVertex3 operator+(CyVertex3&);	// add  for averaging
        Real Dist2(CyVertex3&);			// square of dist
        Real Dist(CyVertex3& w)			// distance
	{
		return sqrt(Dist2(w));
	}

	// polymorphic ops (as friends due to need to commute)
#ifndef NO_IOSTREAM
	friend ostream& operator<<(ostream& s, CyVertex3& v);
#endif
	friend CyVertex3 operator*(CyVertex3&,Real); // scalar mult for weight
	friend CyVertex3 operator*(Real,CyVertex3&);
	//friend CyVertex3 CyVector3::operator+(CyVertex3&);

	// packetizers
	void EncodeToPkt(uchar*, int&);
	void DecodeFromPkt(uchar*, int&);
	int BytesNeededToEncode();
};

//
// CyMatrix class
//
class CyMatrix
{
	friend class CyQuaternion;
protected:
	int mtype;
	Real m[3][4];
public:
	CyMatrix();				// constructor
	CyMatrix(CyMatrix&);			// copy initializer
	void SetQuat(class CyQuaternion& q); // Quat -> CyMatrix (Normalizes Quat)
	void SetQuatN(class CyQuaternion& q);// Normalized Quat -> CyMatrix
	void SetFTU(CyVertex3& from, CyVertex3& to, CyVector3& up);
						// establish c.s. given origin, point
						// on z axis, and up (y) direction
	CyMatrix operator*=(CyMatrix&);	// assignment with composition
	void Print();				// print
	void Print(char *);			// print
	void Ident();
	void Rotx(Real);
	void Roty(Real);
	void Rotz(Real);
	void RotVec( CyVector3 &, Real );
	void Translate(Real x, Real y, Real z);
	//   Here we chain together
	// more and more transformations, muliplying in on the left.
	void ChainRotx(Real);
	void ChainRoty(Real);
	void ChainRotz(Real);
	void ChainRotVec( CyVector3 &, Real );
	void ChainTranslate(Real x, Real y, Real z);
	void ScaleInPlace(Real);
	void ScaleRollPitchYaw(Real,Real,Real,Real);
	void RollPitchYaw(Real,Real,Real);

	// column vector accessors
	CyVector3 U()
	{
		return CyVector3(m[0][0], m[1][0], m[2][0]);
	}
	void U(Real x, Real y, Real z);
	void U( CyVector3 u ) { U( u[X], u[Y], u[Z] ); }
	CyVector3 V()
	{
		return CyVector3(m[0][1], m[1][1], m[2][1]);
	}
	void V(Real x, Real y, Real z);
	void V( CyVector3 v ) { V( v[X], v[Y], v[Z] ); }
	CyVector3 N()
	{
		return CyVector3(m[0][2], m[1][2], m[2][2]);
	}
	void N(Real x, Real y, Real z);
	void N( CyVector3 n ) { N( n[X], n[Y], n[Z] ); }
	CyVertex3 T()
	{
		return CyVertex3(m[0][3], m[1][3], m[2][3]);
	}
	void T(Real x, Real y, Real z);
	void T( CyVertex3 &v );

	// unary ops
	CyMatrix Invert();
	CyMatrix Inverse();

	// binary ops
	CyVector3 operator*(CyVector3&);      // CyMatrix x vector
	CyVertex3 operator*(CyVertex3&);      // CyMatrix x vertex
	CyMatrix operator*(CyMatrix&);      // CyMatrix x CyMatrix

	// packetizers
	void EncodeToPkt(uchar*, int&);
	void DecodeFromPkt(uchar*, int&);
	int BytesNeededToEncode();
};

class CyGMatrix : public CyMatrix	// general matrix (not assumed to be rigid motion)
{
public:
	CyMatrix Invert();
};

/*
	CyRay - defined by a CyVertex3 and a CyVector3
*/
class CyRay
{
	friend class CyPlane;
	CyVertex3 p;
	CyVector3 v;
public:
	CyRay() {}
	CyRay(CyRay&);
	CyRay(CyVertex3& pp, CyVector3& vv)	{ p = pp; v = vv; }
	void Set(CyVertex3& pp, CyVector3& vv)	{ p = pp; v = vv; }

	// packetizers
	void EncodeToPkt(uchar*, int&);
	void DecodeFromPkt(uchar*, int&);
	int BytesNeededToEncode();
};

class CyPlane
{
	CyVertex3 p; // point on plane
	CyVector3 norm; // normal direction
	CyVector3 v1, v2;	// two vectors within the plane
	void compute_vecs();
public:
	CyPlane() {}
	CyPlane(CyPlane&);
	CyPlane(CyVertex3& pp, CyVector3& vv);
	void Set(CyVertex3& pp, CyVector3& vv);
	Bool RayInt(CyRay& r, CyVertex3& res);

	// packetizers
	void EncodeToPkt(uchar*, int&);
	void DecodeFromPkt(uchar*, int&);
	int BytesNeededToEncode();
};

//
// CyQuaternion class
//
class CyQuaternion
{
	friend class CyMatrix;
	Real qscalar;
	CyVector3 v;
public:
	CyQuaternion();									// identity constructor
	CyQuaternion(Real q0, Real q1, Real q2, Real q3);	// initializer
	CyQuaternion(Real scalar, CyVector3 &vec);			// initializer
	CyQuaternion(CyQuaternion&);						// copy initializer
	void SetMatrix( CyMatrix &mat );		// convert matrix to CyQuaternion
	void Print();						// print
	void Print( char * );						// print
	void PrintMat( char * );					// print as a CyMatrix
	CyQuaternion& Normalize();					// normalize the quaternion
	void Set(Real w, Real x, Real y, Real z);	// establish values
	void Set(Real w, CyVector3& v);				// establish values
	Real GetReal( void );				// return real part (scalar)
	CyVector3 GetVector( void );			// return imaginary part (vector)
	CyQuaternion Invert();				// affects this object
	CyQuaternion Inverse();				// does not affect this object
	CyQuaternion operator*( CyQuaternion& );// CyQuaternion * CyQuaternion
	CyQuaternion operator+( CyQuaternion& );// CyQuaternion + CyQuaternion
	CyQuaternion operator-( CyQuaternion& );// CyQuaternion - CyQuaternion
	Real operator%( CyQuaternion & );		// CyQuaternion dot product
	Bool operator==( CyQuaternion& );
	// polymorphic ops (as friends due to need to commute)
	friend CyQuaternion operator*( CyQuaternion &, Real );
	friend CyQuaternion operator*( Real, CyQuaternion & );
	friend CyQuaternion operator/( CyQuaternion &, Real );

	// packetizers
	void EncodeToPkt(uchar*, int&);
	void DecodeFromPkt(uchar*, int&);
	int BytesNeededToEncode();
};

// a CyMatrix interpolation class
//
class CyMatInterp
{
protected:
	CyMatrix mstart, mend, minterp;
	CyQuaternion qstart, qend, qinterp, qi;
	CyVertex3 vstart, vend, vi;
	CyVector3 vinterp;
	Real theta, sintheta;
	Bool interplinear;
public:
	CyMatInterp( CyMatrix &start, CyMatrix &end );

	CyMatrix &Interpolate( Real r );
	void SetType( Bool t ) { interplinear = t; }
	Bool GetType( void ) { return interplinear; }
};

#endif /* LINALGHPP */ //

/*
 * $Log:	linalg.hpp,v $
Revision 1.25  92/03/15  15:47:32  sjd
relaxed the tolerance for REQ from 1e-6 to 1e-3

Revision 1.24  92/03/05  17:27:39  rhh
added vertex - vector operator

Revision 1.23  92/02/18  13:17:52  rhh
added CyMatInterp class (sjd)

Revision 1.22  92/01/01  22:35:56  carlt
added packetization methods for all classes

Revision 1.21  91/12/03  10:26:20  rhh
added Cy prefix to class names

Revision 1.20  91/11/20  13:42:27  carlt
Added CyVertex3::Print()

Revision 1.19  91/11/14  17:30:10  sjd
added CyQuaternion methods, CyMatrix methods.

Revision 1.18  91/10/24  14:31:24  sjd
added samt's RotVec() and ChainRotVec() methods.
This allows rotation about an arbitrary vector.

Revision 1.17  91/10/21  15:58:33  rudy
Allow to add CyVertex3 + CyVertex3 and to multiply scalar times CyVertex3
Purpose is to take weighted average of some points for flocks.

Revision 1.13  91/09/14  15:02:17  rhh
added CyPlane and CyRay stuff

Revision 1.12  91/08/19  18:03:26  rhh
added mtype field, and moved some inline functions to the cpp

Revision 1.11  91/08/12  10:16:19  rudy
Added ScaleInPlace, ScaleRollPitchYaw, and RollPitchYaw.  The
latter two are speedy ways of doing three ChainRots for use
in Tumble method of CyTumbler.

Revision 1.10  91/08/07  10:53:10  sjd
beefed up quaternion class considerably. fixed bugs in CyMatrix::
SetQuat(), and Inverse().

Revision 1.9  91/07/29  10:30:05  rhh
made changes as per meeting 7/22

Revision 1.8  91/05/09  14:56:58  rhh
added overloads for +=, *=, -= on CyVector3, += on CyVertex3, and *= on CyMatrix.

Revision 1.7  91/05/09  12:38:22  carlt
Added matrix scalar multiply

Revision 1.6  91/05/03  09:30:05  sjd
added copyright notice.

Revision 1.5  91/04/12  10:52:57  rhh
added grtypes include.

Revision 1.4  91/02/23  16:47:09  rudy
Adds % for dot product
ChainRotx ChainRoty ChainRotz ChainTranslate
CosSinRotzyx
Frenet

Revision 1.3  91/02/13  10:05:38  rhh
added return value to Unify

Revision 1.2  91/02/10  12:08:52  rhh
added new method to matrix class to print with char *

Revision 1.1  91/01/18  10:50:56  sjd
Initial revision

 */
